<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\BannerRequest;
use App\Banner;
use App\BannerDescription;
use Exception;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\ArticleRequest;

class BannerController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/banners';
    /** View folder */
    private $viewDirectory = 'admin.banners.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('banners.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the pages.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $langId = currentLanguage()->id;
        $query = Banner::orderByDesc('banners.id')
            ->join('banner_descriptions AS sDesc', 'banners.id', 'sDesc.banner_id')
            ->where('sDesc.language_id', $langId)
            ->select([
                'sDesc.title',
                'sDesc.description',
                'banners.*'
            ]);

        if ($request->date_from) {
            $query->whereDate('banners.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('banners.created_at', '<=', $request->date_to);
        }


        if ($request->category_id) {
            $query->where('banners.category_id', $request->category_id);
        }

//        if ($request->title) {
//            $query->where('sDesc.title', 'LIKE', '%'.$request->title.'%');
//        }

//        if ($request->description) {
//            $query->where('sDesc.description', 'LIKE', '%'.$request->description.'%');
//        }


        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $banners= $query->paginate(100);

        return view($this->viewDirectory.'grid',compact('banners'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
//    public function create() :View
//    {
//        $action= route('banners.store');
//        $head= metaFields('banners', 'new', getCurrentLocale()) ?? __('banners.new');
//        $title = $head;
//
//        return view( $this->viewDirectory.'form', get_defined_vars());
//    }

    /**
     * Store a newly created resource in storage.
     *
     * @param BannerRequest $request
     * @return JsonResponse
     */
//    public function store(BannerRequest $request) :JsonResponse
//    {
//        $data = $request->all();
//        $banner = Banner::create($data);
//        $this->saveData($request, $banner->id);
//        session()->flash('message', __('dashboard.saveDone'));
//        return response()->json(['url' => route('banners.index')]);
////        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
//    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Banner $banner
     * @return View
     */
    public function edit(Banner $banner) :View
    {
        $action= route('banners.update', $banner->id);
        $head= metaFields('banners', 'edit', getCurrentLocale()) ?? __('banners.edit');
        $title = $head;

        $query = BannerDescription::where('banner_id', $banner->id)
            ->join('languages', 'languages.id', 'banner_descriptions.language_id')
            ->select(['banner_descriptions.*', 'languages.local']);

        $bannerDescription= $query->get();

        foreach ($bannerDescription as $row) {
            $banner[$row->local]= $row;
        }

        return view( $this->viewDirectory.'form', get_defined_vars() );
    }

    /**
     * Update the specified resource in storage.
     *
     * @param BannerRequest $request
     * @param Banner $banner
     * @return JsonResponse
     */
    public function update(BannerRequest $request, Banner $banner) :JsonResponse
    {

        $data = $request->all();
        $banner->update($data);
        BannerDescription::where('banner_id', $banner->id)->delete();
        $this->saveData( $request, $banner->id );

        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('banners.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param ArticleRequest $request
     * @param int $banner_id
     * @return void
     */

    private function saveData(BannerRequest $request, int $banner_id ):void
    {
        $requestData= $request->all();
        foreach (languages() as $lang) {
            $data = [
                'title'=> $requestData['title_'.$lang->local] ?? null,
                'description'=> $requestData['description_'.$lang->local] ?? null,
                'banner_id'=> $banner_id,
                'language_id'=> $lang->id,
            ];
            BannerDescription::create($data);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $banner = Banner::withTrashed()->find($id);
        if ($banner) {
            if ($banner->deleted_at) {
                $banner->forceDelete();
            } else {
                $banner->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids = $request->ids;
        if ($request->force) {
            Banner::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Banner::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $banner = Banner::withTrashed()->find($id);
        if ($banner){
            $banner->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $banner = Banner::whereIn('id', $request->ids)->onlyTrashed();
        if ($banner){
            $banner->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
